<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';

// Set active page for navigation
$activePage = 'products';

// Check if user is logged in
requireLogin();

// Check if user has permission to view products
// Check if user has any product/inventory-related permissions
$productPermissions = [
    'view_products', 'manage_products', 'view_inventory', 'manage_inventory', 'generate_reports'
];

if (!hasAnyPermission($productPermissions)) {
    $_SESSION['error'] = "You don't have permission to view stock reports.";
    header('Location: dashboard.php');
    exit;
}

if (false && !hasPermission('view_products')) { // Disable the old check
    $_SESSION['error'] = "You don't have permission to access the Stock Report page.";
    header('Location: ../pages/dashboard.php');
    exit;
}

// Get all categories with their products
$stmt = $pdo->query("
    SELECT 
        c.id as category_id,
        c.name as category_name,
        COUNT(p.id) as product_count,
        SUM(p.quantity) as total_quantity,
        SUM(p.quantity * p.cost_price) as total_value
    FROM categories c
    LEFT JOIN products p ON c.id = p.category_id
    GROUP BY c.id, c.name
    ORDER BY c.name
");
$categories = $stmt->fetchAll();

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$storeSettings = [];
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $storeSettings[$row['setting_key']] = $row['setting_value'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stock Report - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        @media print {
            body * {
                visibility: hidden;
            }
            .print-section, .print-section * {
                visibility: visible;
            }
            .print-section {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            .no-print {
                display: none !important;
            }
            .print-header {
                text-align: center;
                margin-bottom: 1rem;
                padding-bottom: 0.5rem;
                border-bottom: 1px solid #e2e8f0;
            }
            .print-header h1 {
                font-size: 1.5rem;
                margin-bottom: 0.25rem;
            }
            .print-header p {
                font-size: 0.75rem;
                margin-bottom: 0.25rem;
            }
            .print-table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 0.75rem;
                font-size: 0.7rem;
            }
            .print-table th, .print-table td {
                padding: 0.25rem 0.5rem;
                border: 1px solid #e2e8f0;
            }
            .print-table th {
                background: #f8fafc;
                font-weight: 600;
            }
            .category-header {
                background: #f0f0f5;
                font-weight: 600;
                padding: 0.25rem 0.5rem;
                border: 1px solid #e2e8f0;
                font-size: 0.8rem;
            }
            .badge {
                padding: 0.15rem 0.35rem;
                font-size: 0.65rem;
            }
            .mb-4 {
                margin-bottom: 0.5rem !important;
            }
            @page {
                margin: 0.5cm;
            }
        }

        .category-card {
            cursor: pointer;
            transition: transform 0.2s;
        }

        .category-card:hover {
            transform: translateY(-2px);
        }

        .product-list {
            display: none;
            margin-top: 1rem;
        }

        .product-list.active {
            display: block;
        }

        .stock-value {
            font-size: 1.25rem;
            font-weight: 600;
        }

        .stock-count {
            font-size: 1.5rem;
            font-weight: 700;
        }

        .badge-stock {
            background: #10b981;
            color: white;
        }

        .badge-low {
            background: #f59e0b;
            color: white;
        }

        .badge-out {
            background: #ef4444;
            color: white;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>

                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2 class="mb-1">Stock Report</h2>
                        <p class="text-muted mb-0">Inventory overview by category</p>
                    </div>
                    <button type="button" class="btn btn-primary" onclick="printReport()">
                        <i class="fas fa-print me-2"></i>Print Report
                    </button>
                </div>

                <!-- Print Section -->
                <div class="print-section" style="display: none;">
                    <div class="print-header">
                        <h1><?php echo htmlspecialchars($storeSettings['store_name'] ?? 'Store Name'); ?></h1>
                        <p><?php echo htmlspecialchars($storeSettings['store_address'] ?? ''); ?></p>
                        <p>Phone: <?php echo htmlspecialchars($storeSettings['store_phone'] ?? ''); ?></p>
                        <p>Stock Report - <?php echo date('Y-m-d H:i:s'); ?></p>
                    </div>

                    <?php foreach ($categories as $category): ?>
                        <div class="mb-4">
                            <div class="category-header">
                                <?php echo htmlspecialchars($category['category_name']); ?> 
                                (<?php echo number_format($category['product_count']); ?>)
                            </div>
                            <table class="print-table">
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Serial</th>
                                        <th>Qty</th>
                                        <th>Remaining</th>
                                        <th>Cost</th>
                                        <th>Total</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    $stmt = $pdo->prepare("
                                        SELECT * FROM products 
                                        WHERE category_id = ? 
                                        ORDER BY name
                                    ");
                                    $stmt->execute([$category['category_id']]);
                                    $products = $stmt->fetchAll();
                                    foreach ($products as $product):
                                    ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($product['name']); ?></td>
                                        <td><?php echo htmlspecialchars($product['serial_no']); ?></td>
                                        <td><?php echo number_format($product['quantity']); ?></td>
                                        <td><?php echo number_format($product['quantity'] - $product['alert_quantity']); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($product['cost_price'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($product['quantity'] * $product['cost_price'], 2); ?></td>
                                        <td>
                                            <?php if ($product['quantity'] == 0): ?>
                                                <span class="badge badge-out">Out of Stock</span>
                                            <?php elseif ($product['quantity'] <= $product['alert_quantity']): ?>
                                                <span class="badge badge-low">Low Stock</span>
                                            <?php else: ?>
                                                <span class="badge badge-stock">In Stock</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <td colspan="3"><strong>Category Total</strong></td>
                                        <td><strong><?php echo number_format($category['total_quantity']); ?></strong></td>
                                        <td colspan="2"></td>
                                        <td><strong><?php echo $currency . ' ' . number_format($category['total_value'], 2); ?></strong></td>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Categories Grid -->
                <div class="row g-4">
                    <?php foreach ($categories as $category): ?>
                        <div class="col-md-4">
                            <div class="card category-card" onclick="toggleProducts(<?php echo $category['category_id']; ?>)">
                                <div class="card-body">
                                    <h5 class="card-title"><?php echo htmlspecialchars($category['category_name']); ?></h5>
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <div>
                                            <div class="text-muted">Products</div>
                                            <div class="stock-count"><?php echo number_format($category['product_count']); ?></div>
                                        </div>
                                        <div>
                                            <div class="text-muted">Total Quantity</div>
                                            <div class="stock-count"><?php echo number_format($category['total_quantity']); ?></div>
                                        </div>
                                        <div>
                                            <div class="text-muted">Total Value</div>
                                            <div class="stock-value"><?php echo $currency . ' ' . number_format($category['total_value'], 2); ?></div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div id="products-<?php echo $category['category_id']; ?>" class="product-list">
                                <div class="card">
                                    <div class="card-body">
                                        <h6 class="card-title mb-3">Products in <?php echo htmlspecialchars($category['category_name']); ?></h6>
                                        <div class="table-responsive">
                                            <table class="table table-sm">
                                                <thead>
                                                    <tr>
                                                        <th>Product</th>
                                                        <th>Qty</th>
                                                        <th>In Stock</th>
                                                        <th>Value</th>
                                                        <th>Status</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php
                                                    $stmt = $pdo->prepare("
                                                        SELECT * FROM products 
                                                        WHERE category_id = ? 
                                                        ORDER BY name
                                                    ");
                                                    $stmt->execute([$category['category_id']]);
                                                    $products = $stmt->fetchAll();
                                                    foreach ($products as $product):
                                                    ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($product['name']); ?></td>
                                                        <td><?php echo number_format($product['quantity']); ?></td>
                                                        <td><?php echo number_format($product['quantity'] - $product['alert_quantity']); ?></td>
                                                        <td><?php echo $currency . ' ' . number_format($product['quantity'] * $product['cost_price'], 2); ?></td>
                                                        <td>
                                                            <?php if ($product['quantity'] == 0): ?>
                                                                <span class="badge badge-out">Out of Stock</span>
                                                            <?php elseif ($product['quantity'] <= $product['alert_quantity']): ?>
                                                                <span class="badge badge-low">Low Stock</span>
                                                            <?php else: ?>
                                                                <span class="badge badge-stock">In Stock</span>
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function toggleProducts(categoryId) {
            const productList = document.getElementById(`products-${categoryId}`);
            productList.classList.toggle('active');
        }

        function printReport() {
            document.querySelector('.print-section').style.display = 'block';
            window.print();
            setTimeout(() => {
                document.querySelector('.print-section').style.display = 'none';
            }, 100);
        }
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 